; kate:  indent-width 4; tab-width 4

;************************************************************************
;*																		*
;* 		O P T R E X   L C D   M O D U L E								*
;*		Code to use a multi-line LCD module orgainzed as N x 16,		*
;*		N x 20, or N x 40 (single E signal)								*
;*																		*
;*		COPYRIGHT AND WARRANTY NOTE										*
;*		This software is copyright (C) 2023, Mark A. Haidekker			*
;*		Software may be used and distributed under the 					*
;*		GNU GPL v 3.0 or later											*
;*		No warranty - use entirely at your own risk						*
;*																		*
;*		See https://www.gnu.org/licenses/gpl-3.0.en.html				*
;*																		*
;************************************************************************
;
; Revision with access RAM and improved initialization (7/2020)


	TITLE "2x16 LCD driver"

	#include p18f13k50.inc
;	Clock: Any

; this module REQUIRES

	EXTERN	waitmsec		; Wait for the number of msec spec'd in WREG
	EXTERN	wait250usec		; Wait for WREG*10 microseconds
	EXTERN	wait10usecp		; Delay W*10 microseconds, better-precision version

; this module PROVIDES

	GLOBAL	LCD_conn_test	; Pulse the LED lines for oscilloscope checking
	GLOBAL	LCD_init		; Initialize the LCD module after reset
	GLOBAL 	LCD_clreol1		; Goto line 1 and clear the line to blank
	GLOBAL	LCD_clreol2		; Goto line 2 and clear the line to blank
	GLOBAL	LCD_clreol		; Clear to eol from current cursor posn
	GLOBAL	LCD_line1		; Place cursor at left of first line
	GLOBAL	LCD_line2		; Place cursor at left of second line
	GLOBAL	LCD_line3		; Place cursor at left of third line
	GLOBAL	LCD_line4		; Place cursor at left of fourth line
	GLOBAL	LCD_cursor_goto	; Place cursor at specified address
	GLOBAL	LCD_home		; Cursor home (same as LCD_line1)
	GLOBAL	LCD_clear		; Clear entire display to blank
	GLOBAL	LCD_write_data	; Write char in WREG to LCD
	GLOBAL	LCD_write_hex	; Write WReg as hex (2 chars)
	GLOBAL	LCD_write_dec		; Write WREG as decimal (2 chars)
	GLOBAL	LCD_write_hex_nyb	; Write WReg as hex (1 char)
	GLOBAL	LCDwrstring		; Write null-terminated string after call instruction


; Some port equates. Note: All physical connections should be
; reflected in this section.

LCD_RS			equ 5		; RC3
LCD_RW			equ 4		; RC4
LCD_E			equ 3		; RC5
LCD_D4			equ	4
LCD_D5			equ	5
LCD_D6			equ	6
LCD_D7			equ	7
LCD_DATA_MASK	equ 0xF0	; Set high for the data bits

LCD_DATA		equ	PORTB		; LCD data port
LCD_LAT			equ	LATB
LCD_TDATA		equ TRISB		; LCD port TRIS
LCD_CTRL		equ	LATC		; LCD control port
LCD_TCTRL		equ TRISC


;
; DATA AREA
;
variables	UDATA_ACS


LCDtemp1	RES	1
LCDtemp2	RES 1


LCD		CODE

;*********************************************************************
;
; OPTREX DISPLAY CONTROL CODE
;
; The Optrex display is controlled in 4-bit mode (meaning:
; only db4-db7 are used); Those are conected to the port specified
; in the EQU section above. Use LCD_DATA and LCD_TDATA.
; LCD control is specified in the EQU section above.
;
;
;*********************************************************************


; Test the connections to the LCD. This function applies square
; waves to each LCD pin with the highest frequency at D4 and
; divided by two in the sequence D5...D7, RW, RS.
; After each cycle, E is clocked (with RW high so we don't run
; into output-output conflicts). At the same time, this function allows
; to test the polled delays. We expect the following frequencies:
;
;		LCD D4		10 KHz
;		LCD D5		5 kHz
;		LCD D6		2.5 kHz
;		LCD D7		1.25 kHz
;		LCD RW		625 Hz				(LCD's pin 5)
;		LCD RS		312 Hz				(LCD's pin 4)
;		LCD E		pulsed high 312Hz	(LCD's pin 6)

LCD_conn_test

		bsf		LCD_CTRL,LCD_E
		bcf		LCD_CTRL,LCD_RS
		bcf		LCD_CTRL,LCD_RW
		bcf		LCD_TCTRL, LCD_E		; Make LCD control ports output
		bcf		LCD_TCTRL, LCD_RW
		bcf		LCD_TCTRL, LCD_RS
		bcf		LCD_TDATA, LCD_D4		; Set 4 data bits to output
		bcf		LCD_TDATA, LCD_D5
		bcf		LCD_TDATA, LCD_D6
		bcf		LCD_TDATA, LCD_D7

LCD_conn_test_0
		movlw	0x3F					; for 6 output lines
		movwf	LCDtemp1
LCD_conn_test_1
		btfss	LCDtemp1,0
		bcf		LCD_DATA,LCD_D4
		btfsc	LCDtemp1,0
		bsf		LCD_DATA,LCD_D4

		btfss	LCDtemp1,1
		bcf		LCD_LAT,LCD_D5
		btfsc	LCDtemp1,1
		bsf		LCD_LAT,LCD_D5

		btfss	LCDtemp1,2
		bcf		LCD_LAT,LCD_D6
		btfsc	LCDtemp1,2
		bsf		LCD_LAT,LCD_D6

		btfss	LCDtemp1,3
		bcf		LCD_LAT,LCD_D7
		btfsc	LCDtemp1,3
		bsf		LCD_LAT,LCD_D7

		btfss	LCDtemp1,4
		bcf		LCD_CTRL,LCD_RW
		btfsc	LCDtemp1,4
		bsf		LCD_CTRL,LCD_RW

		btfss	LCDtemp1,5
		bcf		LCD_CTRL,LCD_RS
		btfsc	LCDtemp1,5
		bsf		LCD_CTRL,LCD_RS

		movlw	.5
		call	wait10usecp
		decfsz	LCDtemp1
		bra		LCD_conn_test_1

		bsf		LCD_CTRL,LCD_RW			; Set LCD to read mode
		bsf		LCD_CTRL,LCD_E			; raise E high
		call	LCD_shortdly			; keep low for a few microsecs
		bcf		LCD_CTRL,LCD_E			; drop E
		bra		LCD_conn_test_0			; and repeat forever






; Perform software reset sequence. Before reset, we can't use the
; busy flag, so we need to delay. These steps are needed:
; Delay 15 msec or more after power up
; Send one nybble 0x03
; Delay 4 msec or more
; Send one nybble 0x03
; Delay 0.1 msec or more
; Send one nybble 0x03
; Delay 0.1 msec or more
; Send command to set 4-bit or 8-bit mode
; Now we can send regular commands (8-bit) through the 
; write-instruction function to set up the unit


LCD_init

		bsf		LCD_CTRL,LCD_E
		bcf		LCD_CTRL,LCD_RS
		bcf		LCD_CTRL,LCD_RW
		bcf		LCD_TCTRL,LCD_E			; Make LCD control ports output
		bcf		LCD_TCTRL,LCD_RW
		bcf		LCD_TCTRL,LCD_RS
		bcf		LCD_TDATA, LCD_D4		; Set 4 data bits to output
		bcf		LCD_TDATA, LCD_D5
		bcf		LCD_TDATA, LCD_D6
		bcf		LCD_TDATA, LCD_D7

		; Let's do it by the book... no LCD_wr_low_nibble, instead keeping
		; the signal levels and only pulsing E.
		; Huh... We need to write to LAT *after* setting TRIS? Since when is that?
		; Perhaps some special function, similar to TRISE?

		bcf		LCD_CTRL, LCD_E			; Set E low (inactive) at the start
		bcf		LCD_CTRL, LCD_RS
		bcf		LCD_CTRL, LCD_RW		; Write mode: Makes LCD data lines input
		bsf		LCD_LAT, LCD_D4			; Apply the soft reset bit pattern 0011
		bsf		LCD_LAT, LCD_D5			; while RW=0 and RS=0
		bcf		LCD_LAT, LCD_D6
		bcf		LCD_LAT, LCD_D7

		movlw	.25					; We begin with E held low
		call	waitmsec			; Wait >15 msec for LCD to settle

		movlw	.25					; We begin with E held low
		call	waitmsec			; Wait >15 msec for LCD to settle

		rcall	LCD_pulse_E			; Pulse E high for >1us to clock in the bit pattern [#1]
		movlw	.6
		call	waitmsec			; Wait >4 msec for LCD to settle

		rcall	LCD_pulse_E			; Pulse E high for >1us to clock in the bit pattern [#2]
		call	wait250usec			; Wait >100 usec for LCD to settle
		
		rcall	LCD_pulse_E			; Pulse E high for >1us to clock in the bit pattern [#3]
		call	wait250usec			; This last wait may or may not be required

		; End soft reset sequence... begin setting up the display
		; Hantronix explains it such that the third write puts the display in 4-bit
		; mode (so far, we have written 12 bits), and that the fourth write (of 4 bits)
		; defines the interface mode. This is either the 4th 8-bit write or the second
		; complete byte in 4-bit mode.

		bcf		LCD_LAT, LCD_D4		; Apply the bit pattern 0010 (set I/F to 4-bit)
		rcall	LCD_shortdly
		rcall	LCD_pulse_E			; Supposedly, we can check for BUSY after this [#4]
									; which also means that we can use 8-bit instruction writes

		; We have one more shaky operation: sending Function Set, which we'll have to send in two
		; 4-bit groups without relying on the busy flag. Execution times are given as 37us.
		; At this point, we should still have the bit pattern 0010. We need to write 0010:1000
		; in two operations with ample microsecond delay in-between.

		rcall	LCD_pulse_E_wait		; Applies pattern 0010 with a 60us wait *before* frobbing E

		bcf		LCD_LAT, LCD_D4			; Apply the lower nybble of the Function Set command,
		bcf		LCD_LAT, LCD_D5			; 1000, while RW=0 and RS=0
		bcf		LCD_LAT, LCD_D6
		bsf		LCD_LAT, LCD_D7
		rcall	LCD_pulse_E_wait		;#5  Wait 60us, then frob the E line...
		movlw	.6						; Give a brief completion delay (spec'd as 37usec
		call	wait10usecp				; so 60us should be plenty).

		; From this point, we continue with regular two-nybble writes that use the BUSY flag

		movlw	b'00001100'			;#6  Display = ON
		rcall	LCD_write_inst		; no cursor, no blink
				
		movlw	b'00000001'			;#7   Display Clear
		rcall	LCD_write_inst

		movlw	b'00000110'			;#8   Entry Mode
		rcall	LCD_write_inst

		movlw	b'10000000'			;DDRAM addresss 0000
		bra		LCD_write_inst		; And done initializing.





;-------------------------------------------------------------
;
; LCD write functions. 
; LCD_write_inst executes the instructions in WREG
; LCD_write_data posts the char in WREG
;
; Timing as follows (one nybble)
;
;     _    ______________________________
; RS  _>--<______________________________
;     _____
; RW       \_____________________________
;                  __________________
; E   ____________/                  \___
;     _____________                ______
; DB  _____________>--------------<______
;

; Note that LCD_wr_low_nibble does not poll the
; busy flag. This is needed during initialization
; when busy is not available (now obsolete)

LCD_clreol1
		call	LCD_line1
		call	LCD_clreol
		bra		LCD_line1

LCD_clreol2
		call	LCD_line2
		call	LCD_clreol
		bra		LCD_line2

LCD_clreol
		movlw	.20
		movwf	LCDtemp1
LCD_clreol_lp
		movlw	A' '
		call	LCD_write_data
		decfsz	LCDtemp1
		bra		LCD_clreol_lp
		return


LCD_cursor_goto					; Set DD addr to WREG (allowable 0x00 to 0x7F,
		iorlw	0x80			; reasonable 0x00 - 0x27 and 0x40 (sic!) - 0x67 for a 2x40 display)
		bra		LCD_write_inst	; Values between .40 and .63 are mapped to .64 and don't work

LCD_line1
		movlw	0x80			; set DD addr to 0 command
		bra		LCD_write_inst

LCD_line2
		movlw	0xC0			; set DD addr to .64 command (start of line 2)
		bra		LCD_write_inst

LCD_line3						; We have a 4-line display, and lines 3 and 4 continue lines 1 and 2
		movlw	(0x80 + .20)	; at an offset of +20 (half of a 40-line display)
		bra		LCD_write_inst

LCD_line4
		movlw	(0xC0 + .20)
		bra		LCD_write_inst

LCD_home
		movlw	0x02			; cursor home command
		bra		LCD_write_inst

LCD_clear
		movlw	0x01			; clear dsp command

LCD_write_inst

		bcf		LCD_CTRL, LCD_RS	; Set RS=0
		bra		LCD_write

LCD_write_data

		bsf		LCD_CTRL, LCD_RS	; Set RS=1

;	Write one byte to LCD

LCD_write
		rcall	LCD_wait_busy		; wait until busy flag clear
		rcall	LCD_wr_high_nibble	; Swap then write (start w/high nyb)
									; Swap again (now low nyb)

LCD_wr_high_nibble

		swapf	WREG

LCD_wr_low_nibble				; (!) This function must keep WREG intact

		bcf		LCD_CTRL, LCD_RW		; set write mode first, *then*
		bcf		LCD_TDATA, LCD_D4		; Set 4 data bits to output (prevent collision)
		bcf		LCD_TDATA, LCD_D5
		bcf		LCD_TDATA, LCD_D6
		bcf		LCD_TDATA, LCD_D7

		bcf		LCD_LAT, LCD_D4		; Clear nibble
		bcf		LCD_LAT, LCD_D5
		bcf		LCD_LAT, LCD_D6
		bcf		LCD_LAT, LCD_D7

		btfsc	WREG, 3			; Transfer nibble
		bsf		LCD_LAT, LCD_D7
		btfsc	WREG, 2
		bsf		LCD_LAT, LCD_D6
		btfsc	WREG, 1
		bsf		LCD_LAT, LCD_D5
		btfsc	WREG, 0
		bsf		LCD_LAT, LCD_D4

										; 'call' and 'nop' give enough of a delay for the lines to settle
		rcall	LCD_pulse_E				; Clock in the data by frobbing E, ensure E=0 on exit.

		bsf		LCD_TDATA, LCD_D4		; Restore the data lines as input to prevent output collision
		bsf		LCD_TDATA, LCD_D5
		bsf		LCD_TDATA, LCD_D6
		bsf		LCD_TDATA, LCD_D7
		bsf		LCD_CTRL, LCD_RW		; ... *then* clear write mode
		return



; Function to set E=1, wait more than 0.5us, and set E=0 again.
; The alternate entry point has a pre-delay of 60usec and destroys W

LCD_pulse_E_wait
		movlw	.6
		call	wait10usecp
LCD_pulse_E
		nop
		bsf		LCD_CTRL, LCD_E
		rcall	LCD_shortdly
		bcf		LCD_CTRL, LCD_E
		return




;-------------------------------------------------------------
;
; LCD read functions. 
; LCD_read_reg reads the address register with busy flag
; LCD_read_data reads the display RAM
; In both cases, the data word is returned in WREG (LCDtemp2 is used and gets destroyed)
;
; Timing as follows (one nybble)
;     _____    _____________________________________________________
; RS  _____>--<_____________________________________________________
;               ____________________________________________________
; RW  _________/
;                  ____________________      ____________________
; E   ____________/                    \____/                    \__
;     _________________                __________                ___
; DB  _________________>--------------<__________>--------------<___
;


LCD_read_reg

		bcf		LCD_CTRL, LCD_RS	; Set RS=0
		bra		LCD_read

LCD_read_data

		bsf		LCD_CTRL, LCD_RS	; Set RS=1

LCD_read
		bsf		LCD_TDATA, LCD_D4		; Set 4 data bits to input
		bsf		LCD_TDATA, LCD_D5
		bsf		LCD_TDATA, LCD_D6
		bsf		LCD_TDATA, LCD_D7

		bsf		LCD_CTRL, LCD_RW	; Then set read mode
		rcall	LCD_shortdly		; Short delay

		bsf		LCD_CTRL, LCD_E		; Setup to clock data
		rcall	LCD_shortdly		; Short delay

		movf	LCD_CTRL, W			; Get data (high nibble)
		andlw	LCD_DATA_MASK		; Mask away unused half of the port
		movwf	LCDtemp2
		swapf	LCDtemp2

		bcf		LCD_CTRL, LCD_E		;  clock data low
		rcall	LCD_shortdly		; Short delay

		bsf		LCD_CTRL, LCD_E		;  clock data high
		rcall	LCD_shortdly		; Short delay

		movf	LCD_CTRL, W			; Get data (now low nibble)
		andlw	LCD_DATA_MASK		; Mask away unused half of port, again
		iorwf	LCDtemp2, W			; Merge with high nyb

		bcf		LCD_CTRL, LCD_E		;  clock data low, finished reading
		return						; must exit with E=0



; Wait until LCD signals that it's no longer busy
; must preserve WREG _and_ LCD_RS
; Both bits get intermediate storage in LCDtemp2

LCD_wait_busy

		clrf	LCDtemp2
		btfsc	LCD_CTRL, LCD_RS
		bsf		LCDtemp2, 0				; transfer PORTE0 to LCDtemp2
		bcf		LCD_CTRL, LCD_RS		; Set RS=0
		bsf		LCD_TDATA, LCD_D4		; Set 4 data bits to input
		bsf		LCD_TDATA, LCD_D5
		bsf		LCD_TDATA, LCD_D6
		bsf		LCD_TDATA, LCD_D7

		bsf		LCD_CTRL, LCD_RW		; Then set read mode
		rcall	LCD_shortdly			; Short delay

LCD_wait_busy_loop
		bsf		LCD_CTRL, LCD_E			; Setup to clock data
		rcall	LCD_shortdly			; Short delay

		btfsc	LCD_DATA, LCD_D7		; transfer the busy flag bit
		bsf		LCDtemp2, 1				; (high mode)
		btfss	LCD_DATA, LCD_D7		; transfer the busy flag bit
		bcf		LCDtemp2, 1				; (low mode)

		bcf		LCD_CTRL, LCD_E			;  Now perform a dummy read of the
		rcall	LCD_shortdly			; low nybble
		bsf		LCD_CTRL, LCD_E
		rcall	LCD_shortdly
		bcf		LCD_CTRL, LCD_E			;  clock data low, finished reading

		btfsc	LCDtemp2, 1				; test the busy bit
		bra		LCD_wait_busy_loop

		btfsc	LCDtemp2,0
		bsf		LCD_CTRL, LCD_RS		; transfer the old RS bit
		btfss	LCDtemp2,0
		bcf		LCD_CTRL, LCD_RS
		return							; Must exit with E==0 (see above)


; NOP-pile delay functions. Crude, but they don't affect W.
; The shorter LCD_shortdly has 22 NOPs. 
; At 40MHz clock (10 MIPS), the delay is rougly 2.5 microseconds
; (with call & return). 

LCD_shortdly
		nop
		nop
		nop
		nop
		nop
		nop
		nop
		nop
		nop
		nop
		nop
		nop
		nop
		nop
		nop
		nop
		nop
		nop
		nop
		nop
		nop
		return





;-------------------------------------------------------------
;
; Higher-level LCD write functions:
; Hex, decimal, and string


; write one hex byte. Preserves WREG by means of LCDtemp1

LCD_write_hex

		movwf	LCDtemp1
		swapf	WREG
		rcall	LCD_write_hex_nyb
		movf	LCDtemp1, W
		rcall	LCD_write_hex_nyb
		movf	LCDtemp1, W
		return


; Write WREG as dec number - this function is a small version
; of x_to_bcd in INTMATH. Just one byte, therefore much
; easier to call, although not the fastest. Note: W must not exceed 99.
; Does not preserve WREG

LCD_write_dec

		movwf	LCDtemp1		; first we must BCD encode W
		clrf	LCDtemp2		; This will later have the BCD tens
LCD_write_dec_0
		movlw	0x0A	
		subwf	LCDtemp1, F
		bn		LCD_write_dec_1	; Done subtracting. Now go print.
		incf	LCDtemp2
		bra		LCD_write_dec_0
LCD_write_dec_1
		movlw	0x0A
		addwf	LCDtemp1, F		; Now contains the low BCD byte
		movf	LCDtemp2, W		; the high byte
		call	LCD_write_hex_nyb
		movf	LCDtemp1, W

LCD_write_hex_nyb

		andlw	0x0f
		movwf	LCDtemp2
		movlw	0x0a
		cpfslt	LCDtemp2
		addlw	0x07
		addwf	LCDtemp2, W
		addlw	0x26
		bra		LCD_write_data


; LCDwrstring uses the return stack to display a in-code string.
; When this function is called, the return address (caller+2)
; is in TOS (TOSU/TOSH/TOSL). We grab data from [TOS] until there
; is a zero. All non-zero chars are sent to LCD; Then, the TOS is
; incremented past the finising zero so we return beyond the string.
; Note that all data is 16-bit, so any string must have an EVEN
; length or the program will die (or maybe not because those
; zeroes will be executed as NOP).
; Also, to prevent change of TOS during reading or writing TOS,
; we need to disable interrupts!

LCDwrstring

		bcf		INTCON, GIE			; No interrupts while we read the stack!!
		movff	TOSU,TBLPTRU
		movff	TOSH,TBLPTRH
		movff	TOSL,TBLPTRL
		bsf		INTCON, GIE

wrstring_next
		tblrd	*+
		movf	TABLAT,W
		bz		exit_wrstring
		rcall	LCD_write_data
		bra		wrstring_next

exit_wrstring					; TBLPTR now points _beyond_ the trailing zero

		bcf		INTCON, GIE		; No interrupts while we write the return address!
		tblrd*+					; Dummy read to skip the 2nd zero
		movf	TBLPTRU,W
		movwf	TOSU
		movf	TBLPTRH,W
		movwf	TOSH
		movf	TBLPTRL,W
		movwf	TOSL
		bsf		INTCON, GIE
		return



		end	


;*********************************************************************


