; kate:  indent-width 4; tab-width 4

;************************************************************************
;*																		*
;* 		M R I   P R O J E C T :  I N D U C T A N C E   M E T E R		*
;*		==========================================================		*
;*																		*
;*		Version 1.0 	 												*
;*																		*
;*																		*
;*		COPYRIGHT AND WARRANTY NOTE										*
;*		This software is copyright (C) 2023, Mark A. Haidekker			*
;*		Software may be used and distributed under the 					*
;*		GNU GPL v 3.0 or later											*
;*		No warranty - use entirely at your own risk						*
;*																		*
;*		See https://www.gnu.org/licenses/gpl-3.0.en.html				*
;*																		*
;************************************************************************

	TITLE "MRI Project: Inductance Meter"

      #include <p18f13k50.inc>

;	#include <p18f4520.inc>
;	Clock: 40 MHz external oscillator
;	(in a pinch, we could configure the device for 32MHz via PLL)


; -----------------------------------------------------------------------
;
; Configuration bits.
; See /usr/share/gputils/header/p18f13k50.inc, line 1188ff for details.
;

;	CONFIG	CPUDIV=NOCLKDIV, USBDIV=OFF, FOSC=ECH, PLLEN=OFF, PCLKEN=ON, FCMEN=OFF, IESO=ON, PWRTEN=ON, BOREN=ON, BORV=27, WDTEN=OFF, MCLRE=ON, STVREN=ON, LVP=OFF

; Use the following line when running the 40MHz external oscillator

	CONFIG	FOSC=ECH, CPUDIV=NOCLKDIV, USBDIV=OFF, PLLEN=OFF, PCLKEN=ON, FCMEN=OFF
	CONFIG	IESO=ON, PWRTEN=ON, BOREN=ON, BORV=27, WDTEN=OFF, MCLRE=ON, STVREN=ON, LVP=OFF


;************************************************************************
;
; I/O usage table for PIC18F13k50
;
;
;	RA0		USB D+ and ICD2
;	RA1		USB D- and ICD2
;	(RA2)	(not present)
;	RA3		MCLR
;	RA4		A	AN3: Signal Level Input
;	RA5		(Clki)
;
;	RB4		IO	LCD D4
;	RB5		IO	LCD D5
;	RB6		IO	LCD D6
;	RB7		IO	LCD D7
;
;	RC0		I	INT0, Pushbutton, active high
;	RC1		I	Range switch, 1nF, active high
;	RC2		I	Range switch, 300pF, active high
;	RC3		O	LCD RS
;	RC4		O	LCD R/W
;	RC5		O	LCD E
;	RC6		I	TMR1 Oscillator input
;	RC7		O	Status LED
;
;
;
;************************************************************************
;
; TIMER USE
;
;	TMR0		100Hz heartbeat timer (ISR)
;	TMR1		Counter to determine frequency
;	TMR2		
;	TMR3		Polled delays
;
; FSR use
;
;	FSR0
;	FSR1
;	FSR2
;
; RAM use -- the 13k50 has only 512 bytes of RAM
;
;	Access RAM: Bytes 0...95 from Bank 0 (GPRAM), Bytes 96 to 255 from Bank15 (SFR)
;	Non-access RAM
;		Bank 0		00:60 to 00:FF	(160 bytes) = GPRAM
;		(Bank1		01:00 to 01:FF		GPRAM in the 18F14k50 only)
;		Bank 2		02:00 to 02:FF	shared USB DPRAM and GPRAM
;	
;
;************************************************************************





; 1. Math module

	EXTERN	xreg0,xreg1,xreg2,xreg3,xreg4
	EXTERN	yreg0,yreg1,yreg2,yreg3
	EXTERN	zreg0,zreg1,zreg2,zreg3
	EXTERN	treg0,treg1,treg2,treg3
	EXTERN	dreg0,dreg1,dreg2,dreg3

	EXTERN	init_dreg			; Set decade register to 1
	EXTERN	clear_xreg			; Zero X register
	EXTERN	clear_x_to_W
	EXTERN	copyxy				; Copy Xreg into Yreg
	EXTERN	copyxz				; Copy Xreg into Zreg
	EXTERN	copyzx				; Copy zreg into xreg
	EXTERN	copydx				; Copy Dreg into Xreg
	EXTERN	copyxt				; Copy Xreg into Treg
	EXTERN	copyxd				; Copy Xreg into Dreg
	EXTERN	swapxy				; Exchange X and Y
	EXTERN	abs_x				; Compute absolute of xreg ( abs(x) -> x )
	EXTERN	chs_x				; Change sign of xreg ( -x -> x )
	EXTERN	clamp_x				; Clamp negative values in X to zero
	EXTERN	expand_sign			; Expand sign of 16-bit X to 32 bits
	EXTERN	xtest0				; Test if x==0, Z if so
	EXTERN	xplusy				; Add yreg to xreg:	y + x -> x
	EXTERN	xplusz				; Add zreg to xreg:	z + x -> x
	EXTERN	tplusx				; Add xreg to treg:	t + x -> t
	EXTERN	yminusx				; subtract xreg from yreg: y - x -> x
	EXTERN	cmpxy				; Test if xreg >= yreg through dummy x-y. C=1 if yes
	EXTERN	scmpxy				; Same as above, but for signed ints
	EXTERN	cmpyz				; Test if yreg >= zreg through dummy y-z. C=1 if yes
	EXTERN	lshiftx				; Left shift XREG (both signed and unsigned)
	EXTERN	rshiftx				; Right shift xreg (signed)
	EXTERN	xtimesw				; Multiply Xreg times W (fast harware mul)
	EXTERN	sxtimesw			; Multiply signed Xreg times W (fast harware mul)
	EXTERN	xtimes10			; Multiply Xreg times 10 (fast harware mul)
	EXTERN	mulxy				; Multiplication x*y -> x
	EXTERN	smulxy				; Signed Multiplication x*y -> x
	EXTERN	subyz				; Subtract Zreg from Yreg: y=y-z
	EXTERN	divyx				; Division y/x -> x
	EXTERN	sdivyx				; Signed Division y/x -> x
	EXTERN	x_to_bcd			; Convert xreg into BCD
	EXTERN	intsqrt


; 2. LCD module

	EXTERN	LCD_conn_test
	EXTERN	LCD_init		; Initialize the LCD module after reset
	EXTERN 	LCD_clreol1		; Goto line 1 and clear the line to blank
	EXTERN	LCD_clreol2		; Goto line 2 and clear the line to blank
	EXTERN	LCD_clreol		; Clear to eol from current cursor posn
	EXTERN	LCD_line1		; Place cursor at left of first line
	EXTERN	LCD_line2		; Place cursor at left of second line
	EXTERN	LCD_line3		; Place cursor at left of third line
	EXTERN	LCD_line4		; Place cursor at left of fourth line
	EXTERN	LCD_cursor_goto	; Place cursor at specified address
	EXTERN	LCD_home		; Cursor home (same as LCD_line1)
	EXTERN	LCD_clear		; Clear entire display to blank
	EXTERN	LCD_write_data	; Write char in WREG to LCD
	EXTERN	LCD_write_hex	; Write WReg as hex (2 chars)
	EXTERN	LCD_write_dec		; Write WREG as decimal (2 chars)
	EXTERN	LCD_write_hex_nyb	; Write WReg as hex (1 char)
	EXTERN	LCDwrstring		; Write null-terminated string after call instruction


; This modules SHARES:

	GLOBAL	shortwait, shortwait5
	GLOBAL	wait250msec, waitmsec, wait10usecp, wait250usec




;*****************************************************************************

; General equates

TMR0PRELOAD			equ		(.65536 - .12500)		; for a 10ms heartbeat interrupt

; The following are the constants for inductance computation for the
; three ranges, 330pF, 1nF, 4.7nF (L-H for easier access)
; intended to yield inductance in units of 10nH.
; The wire jungle prototype and the PCB version use a different divider.
; Uncomment the set of constants for the device,
;
; Wire jungle version:

;FCONST_RNG1_H		equ		0x1173			; Range 1, 330pF,  292,810,335
;FCONST_RNG1_L		equ		0xEE5F

;FCONST_RNG2_H		equ		0x05C2			; Range 2, 1nF,	    96,627,411
;FCONST_RNG2_L		equ		0x6AD3

;FCONST_RNG3_H		equ		0x0139			; Range 3, 4.7nF,   20,559,024
;FCONST_RNG3_L		equ		0xB4B0

; PCB version

;FCONST_RNG1_H		equ		0x45CF			; Range 1, 330pF, 1 171 241 340
;FCONST_RNG1_L		equ		0xB97C

;FCONST_RNG2_H		equ		0x1709			; Range 2, 1nF,	    386 509 642
;FCONST_RNG2_L		equ		0xAB4A

;FCONST_RNG3_H		equ		0x04E6			; Range 3, 4.7nF,    82 236 094
;FCONST_RNG3_L		equ		0xD2BE

; With calibration

FCONST_RNG1_H		equ		0x2749			; Range 1, calibrated 586pF, 659,134,212
FCONST_RNG1_L		equ		0x9704

FCONST_RNG2_H		equ		0x12ED			; Range 2, calibrated 1.22nF, 317,548,937
FCONST_RNG2_L		equ		0x6989

FCONST_RNG3_H		equ		0x051C			; Range 3, calibrated 4.51nF, 85,768,529
FCONST_RNG3_L		equ		0xB951

; Inductor offset (nH divided by 10), from the wires

IND_OFFS			equ		.35



; Port equates
;
; Port A
; (no general functions)

; Port B
; (no general functions, merely LCD data lines)

; Port C

PUSHBUTTON		equ		0			; Pushbutton input
RANGESEL1		equ		2			; Range selector, 1nF
RANGESEL0		equ		1			; Range selector, 300pF
									; If both are low, the switch is in the 4.7nF position
DEBUG			equ		7			; For now, debug pin
LED				equ		7			; ... or LED (CENGR PCBs)


; Bit equates

RS_DAVAIL		equ		0
RS_DSPREQ		equ		1
RS_CONNECTED	equ		2

;
; Non-access RAM equates (tables etc)
;




;
; DATA AREA
; Variables stored consecutively in data memory
; These start at 0x60 in Bank 0 (non-access RAM)
;


accessvar	UDATA_ACS

iwreg		RES 1		; WREG save for interrp
istatus		RES 1		; STATUS save for interrp
ibsr		RES 1		; BSR save for interrp
itemp		RES 1		; Temp used in intr AND NOWHERE ELSE




variables	UDATA


resetreg	RES	1
temp1		RES	1
temp2		RES	1
runstatus	RES	1		; Runstatus bits (PLL run status)
sysstatus	RES	1		; System (overall) status
tencounter	RES	1		; To get down to 100ms in ISR
dspycnt		RES	1		; Display counter/delay
tmr1ovfl	RES	1		; Overflow byte for TMR1, set by ISR. Makes TMR1 effectively 24-bit wide
count_l		RES	1		; Copy of TMR1 after read-out
count_h		RES	1
count_u		RES	1
osc_lvl_l	RES	1		; Analog input, oscillator amplitude
osc_lvl_h	RES	1








;************************************************************************

;
; VECTORS
;


STARTUP CODE
	org		00h
 	goto	Reset_ent
	
	org		08h
	goto	Intrp_ent	; High-priority interrupt

	org 	018h
	goto	Low_intrp_ent	; Low-priority interrupt

;***************************************************************************
; Program code area
;


MAIN 	CODE

;
; Entry point for reset

Reset_ent


		clrf	WREG
initdly
		decfsz	WREG
		bra		initdly


; save the reset register (for later examination of the cause of a RESET)
; then initialize it & disable BOR with SBOREN

		movf	RCON, W
		andlw	B'00011111'			; mask the reset flags
		movwf	resetreg
		bcf		RCON, SBOREN		; software-disable BOR
		bsf		RCON, RI			; clear soft-reset flag
		bsf		RCON, POR			; clear power-on reset flag	
		bsf		RCON, BOR			; clear BOR flag

; Set up the clock. Internal 8-MHz, but using as def'd in CONFIG1

		movlw	B'01101000'
		movwf	OSCCON

;
; Prepare ports & port directions
;

		clrf	LATA
		clrf	LATB
		clrf	LATC
		setf	TRISA				; Mostly inputs, analog
		setf	TRISB				; 4 LCD I/O lines, set in LCD module
		movlw	B'01000111'			; Port C: mixed inputs & outputs
		movwf	TRISC



; Turn off features that we don't need: Comparators, SR

		bcf		CM1CON0, C1ON		; CM1 globally off
		bcf		CM1CON0, C1OE		; and specifically off the I/O ports
		bcf		CM2CON0, C2ON		; and the same for CM2
		bcf		CM2CON0, C2OE

		bcf		SRCON0, SRLEN		; Disable SR (should be disabled by reset)
									; Oh, and make sure the LVP config bit is clear


; Configure the ADC. We need one channel, RA4/AN3


		movlw	B'00001101'			; A/D is on, select AN3 for the time being
		clrf	ADCON0
		clrf	ADCON1				; A/D references are VDD, VSS
		movlw	B'10011110'			; Clock/64; 6 Tad
		movwf	ADCON2
		movlw	B'00001000'			; ANSEL<3>, presently the only analog channel defined
		movwf	ANSEL				; all-digital I/O on Port B and C
		movlw	B'00000000'
		movwf	ANSELH				; No more analog bits


;
; Clear variables
;

		clrf	runstatus
		clrf	sysstatus
		clrf	count_l
		clrf	count_h
		clrf	count_u
		movlw	.20
		movwf	tencounter				; Reset the counter-to-ten
		clrf	dspycnt
		clrf	tmr1ovfl


; TIMER SETUP
;
; TMR0
;       TMR0 is used for the interrupt subsystem and heartbeat
;		We use an 1:8 prescaler, and TMR0 is clocked at 1.25 MHz (800ns)
;		The 10ms heartbeat requires 12500 ticks (see TMR0PRELOAD)

		movlw	B'10000010'
		movwf	T0CON

; TMR1
;		TMR1 serves as frequency counter, has external clock input
;		at T1OSCI (RC6).
;		The signal at T1OSCI is the oscillator frequency divided by 64,
;		and we can reasonably expect th remain below 10MHz, which means that
;		TMR1 can be clocked with as much as 156kHz. In this worst case,
;		TMR1 would roll over after 420ms. But who says we need to count
;		over a full second?
;		Also, we do not need the internal TMR1 oscillator. No synching, and
;		TMR1 is not the main clock source.

		movlw	B'10000111'
		movwf	T1CON

; TMR3
;		TMR3 is used for polled delays. It runs off of the main clock
;		with a 1:2 prescaler (5MHz) so that one increment is 200ns.
;		A 10us wait therefore requires 50 ticks; 250us done with 1250 ticks.

		movlw	B'10011001'
		movwf	T3CON


; Set up the interrupts.
;
; No priorities needed. We have
;		TMR0		Heartbeat
;		TMR1		Event counter, interrupts on overflow


		bcf		INTCON, TMR0IF			; clear possibly pending TMR0 interrupt
		bsf		INTCON, TMR0IE			; and enable TMR0 interrupts 
		bcf		PIR1, TMR1IF			; Clear potentially pending interrupt
		bcf		PIE1, TMR1IE			; Keep TMR1 interrupt disabled for now
		bcf		PIR1, ADIF
		bcf		PIE1, ADIE
		bcf		RCON, IPEN				; Disable interrupt priorities
		bsf		INTCON, PEIE			; Enable peripheral interrupts
		bsf		INTCON, GIE				; Enable all interrupts



; How about trying the LCD next?

;		call	LCD_conn_test
		call	LCD_init
		call	LCDwrstring
		data	"M.A.HAIDEKKER'S ", 0
		call	LCD_line2
		call	LCDwrstring
		data	"INDUCTOR METER", 0
		call	wait250msec


;********************************************************************************
; 
;  #        ###     ###    ####   
;  #       #   #   #   #   #   #  
;  #       #   #   #   #   #   #  
;  #       #   #   #   #   ####   
;  #       #   #   #   #   #      
;  #       #   #   #   #   #      
;  #####    ###     ###    #      
;
;
;********************************************************************************




loopforever
		nop
		nop
		nop
		nop

		btfsc	runstatus, RS_DAVAIL
		rcall	loop_process_data

		btfsc	runstatus, RS_DSPREQ
		rcall	loop_display

		nop
		bra		loopforever



; This is called whenever data from TMR1 were read out.
; This function adds them up to cntavg_<l:h> up to four times,
; then sets the display request

loop_process_data
		bcf		runstatus, RS_DAVAIL

		; count_<l:h> has 200ms worth of counts which is 1/5 of the frequency
		; that is in turn divided by 64. Overall, this value, times 320, 
		; is the real frequency

		movf	count_l, W
		movwf	xreg0
		movf	count_h, W
		movwf	xreg1
		clrf	xreg2
		clrf	xreg3
		bsf		runstatus, RS_DSPREQ
		return



; Display function, includes the math.

loop_display
		bcf		runstatus, RS_DSPREQ
		call	LCD_clear

		; Display depending on button press

		btfsc	PORTC, PUSHBUTTON
		bra		loop_dsp_raw				; Raw data if pressed

		; Test if an inductor is connected

		btfss	runstatus, RS_CONNECTED
		bra		loop_dsp_connfail

		call	xsquare
		call	calc_induct
		call	display_inductance
		bra		loop_dsp_line2

		; Low amplitude: No coil connected, likely.

loop_dsp_connfail
		call	LCDwrstring
		data	"CONNECTED?  ", 0
		bra		loop_dsp_line2

		; When the pushbutton is pressed, up-calculate the actual frequency
		; and display it in kHz with one decimal. 200ms count is in xreg.
		; count x5 gives Hz, but scaled by the HW divider. Another x32 
		; (wire jungle PCB: x64) for a total total of of x 160 (x320 wire jungle) 
		; is the oscillator frequency in Hz. To display in kHz with
		; one digit, we need to divide by 1000 -- but then we can also multiply
		; with 16 (32) and divide by 10 (kHz with one decimal). Or not divide at all
		; and have two decimals.

loop_dsp_raw
		movlw	.16					; Was 32 for wire jungle prototype
		call	xtimesw				; 200ms: x5 for Hz, x64 to revert the prescaler
		call	print.x.twodecs
		call	LCDwrstring
		data	"kHz   ", 0

		; In Line 2, display the range select value
		;  1234567890123456
		; "RNG: 300pF C=125

loop_dsp_line2
		call	LCD_line2
		call	LCDwrstring
		data	"RNG:", 0
		btfsc	PORTC, RANGESEL0
		bra		loop_dsp_rng300
		btfsc	PORTC, RANGESEL1
		bra		loop_dsp_rng1000
		call	LCDwrstring
		data	"4.7nF C=", 0
		bra		loop_dsp_lvl
loop_dsp_rng300
		call	LCDwrstring
		data	"330pF C=", 0
		bra		loop_dsp_lvl
loop_dsp_rng1000
		call	LCDwrstring
		data	"1nF   C=", 0

		; and print the signal peak level (high byte only)

loop_dsp_lvl
		rrcf	osc_lvl_h, W
		movwf	xreg1
		rrcf	osc_lvl_l, W
		movwf	xreg0
		rrcf	xreg1, F			; This will now be zero
		rrcf	xreg0, F
		clrf	xreg1				; Still, kill potential bits shifted in from C
		clrf	xreg2
		clrf	xreg3
		goto	print.x


; 16-bit xreg squared, with hardware multiplier -- fast.
; Uses yreg.
; Explicit: x0*x0 + 2*256*(x1*x0) + 65536*(x1*x1)
; We use the 200ms-count, but divided by 16. So after the square operation,
; we can also discard the LSByte, rounding.

xsquare
		movf	xreg0, W
		mulwf	xreg0
		movff	PRODL, yreg0
		movff	PRODH, yreg1
		movf	xreg1, W
		mulwf	xreg1
		movff	PRODL, yreg2
		movff	PRODH, yreg3
		movf	xreg0, W

		; Mixed term, add twice instead of shifting
		; because there is no guarantee that PROD<L:H> is below 16 bits

		mulwf	xreg1
		movf	PRODL, W
		addwf	yreg1, F
		movf	PRODH, W
		addwfc	yreg2, F
		movlw	0x00
		addwfc	yreg3, F

		bsf		STATUS, C			; The second addition includes rounding
		movf	PRODL, W
		addwfc	yreg1, F
		movf	PRODH, W
		addwfc	yreg2, F
		movlw	0x00
		addwfc	yreg3, F
		movff	yreg1, xreg0		; Result goes back to Xreg
		movff	yreg2, xreg1		; but the LSByte gets dropped
		movff	yreg3, xreg2
		clrf	xreg3
		return

; x squared with full intmath use (not used in code)

xsquared_lib
		call	copyxy
		call	mulxy
		movlw	0x00
		bsf		STATUS, C			; Rounding operation
		addwfc	xreg1, F
		addwfc	xreg2, F
		addwfc	xreg3, F

		; Discard the LSByte

		movff	xreg1, xreg0
		movff	xreg2, xreg1
		movff	xreg3, xreg2
		clrf	xreg3
		return


;----------------------------------------------
;
; Inductance calculation.
; We have frequency squared in yreg
; The general formula is
;
;			L = 1/( (2 pi f)^2 C)
;
; for which we can precompute 1/(4 pi^2 C)
; Important: The frequency is scaled, and the scaling factor
; differs between the first version ("wire jungle") and the PCB version.
;
;		* sampled every 200ms
;		* hardware-prescaled by 1/32 (wire jungle: 1/64)
; so the true frequency per RS_DSPREQ is
;		count * 5 * 32 = count*160 (in Hz)
;		count * 5 * 64 = count*320 (in Hz) -- wire jungle
; and the constants
;		* Range 1 (330pF): 76,758,473
;		* Range 2 (1nF):   25,330,296
;		* Range 3 (4.7nF):  5,389,425
; Dividing these constants by f^2 yields the inductance in H.
; Since these constants need to be divided by f^2, we can integrate
; the scale factor of 160^2 (320^2) *and* multiply with 1e7 to obtain inductance
; in units of 0.1uH. This value is far too large, so we use the f^2 value
; calculated above that is divided by 16^2 and divide this constant by 256, too.
; Overview: The constant emerges from
;
;  PCB version								Wire jungle version
;
;	   1          10^7      1					  1          10^7      1
;  ---------- * ------- * -----				  ---------- * ------- * -----
;   4 pi^2 C     160^2     256				   4 pi^2 C     320^2     256
;
; PCB version
;		* Range 1 (330pF):  117 124 134				0x 06FB:2C25
;		* Range 2 (1nF):     38 650 964   			0x 024D:C454
;		* Range 3 (4.7nF):    8 223 609				0x 007D:7B79
;
; Wire jungle version
;		* Range 1 (330pF): 29 281 034				0x 01BE:CB0A
;		* Range 2 (1nF):    9 662 741				0x 0093:7115
;		* Range 3 (4.7nF):  2 055 902				0x 001F:5EDE
;
; in wihch case dividing by our internal f^2 yields the inductance in 1/10 uH.
;
; NB: We can in all cases also multiply by 16 and use f^2 scaled down by 16 instead of 256.
; This gets us close to a second nH digit. Or: Using the f^2 scaled as presently done (by 256),
; we can use for units of 1/100 uH
;
;						   PCB version						Wire jungle version
;		* Range 1 (330pF): 1 171 241 340 (0x45CF:B97C)		292 810 335 (0x1173:EE5F)
;		* Range 2 (1nF):     386 509 642 (0x1709:AB4A)		 96 627 411	(0x05C2:6AD3)
;		* Range 3 (4.7nF):    82 236 094 (0x04E6:D2BE)       20 559 024	(0x0139:B4B0)
;
; The latter constants are given as equates at the top of this program.



calc_induct

		; Load yreg with the range-dependent constant

		btfsc	PORTC, RANGESEL0
		bra		calc_induct_rng300
		btfsc	PORTC, RANGESEL1
		bra		calc_induct_rng1000

		; 4.7nF range here.  5,389,425 (scaled 20,559,024 = 0x 0139:B4B0)

		movlw	high FCONST_RNG3_H
		movwf	yreg3
		movlw	low FCONST_RNG3_H
		movwf	yreg2
		movlw	high FCONST_RNG3_L
		movwf	yreg1
		movlw	low FCONST_RNG3_L
		movwf	yreg0
		bra		calc_induct_proper

		; 300pF range here. 84,434,320 (scaled 292,810,335 = 0x 1173:EE5F)

calc_induct_rng300
		movlw	high FCONST_RNG1_H
		movwf	yreg3
		movlw	low FCONST_RNG1_H
		movwf	yreg2
		movlw	high FCONST_RNG1_L
		movwf	yreg1
		movlw	low FCONST_RNG1_L
		movwf	yreg0
		bra		calc_induct_proper

		; 1nF range here. 25,330,296 (scaled 96,627,411 = 0x 05C2:6AD3)

calc_induct_rng1000
		movlw	high FCONST_RNG2_H
		movwf	yreg3
		movlw	low FCONST_RNG2_H
		movwf	yreg2
		movlw	high FCONST_RNG2_L
		movwf	yreg1
		movlw	low FCONST_RNG2_L
		movwf	yreg0

		; Join paths again. Divide the constant by f^2 (in xreg)
		; then subtract the wire inductance offset.

calc_induct_proper
		call	divyx		; Integer 1/100 mH. Remainder in yreg.
		movlw	IND_OFFS	; Wire inductance, units of 10nH, 1/100mH
		subwf	xreg0, F
		movlw	0x00
		subwfb	xreg1, F
		subwfb	xreg2, F
		subwfb	xreg3, F
		return


; Display the inductance in xreg. We have two formats, depending on the range.
; The value is in 10nH. Use this value as-is and print with two digits up to
; 999 x 10nH (as 9.99uH). Above 999, print with one digit, for which we require
; a rounding divide-by-10.
; NB: If the inductance is negative display "invalid"

display_inductance
		btfsc	xreg3, 7		; Negative?
		bra		dsp_ind_too_small
		movlw	low .1000
		movwf	yreg0			; cmpxy is a >= comparison
		movlw	high .1000
		movwf	yreg1
		clrf	yreg2
		clrf	yreg3
		call	cmpxy			; dummy x-y. C=1 if x >= y (x >= 1000)
		bc		display_induct_uh

		; Two digits here. Displayed as 1.34

		call	print.x.twodecs
		bra		display_induct_units

		; After subtracting the offset, negative value

dsp_ind_too_small
		call	LCDwrstring
		data	"TOO SMALL?  ",0
		return

		; Display with one digit. For this, we can use print.x.10, but that
		; needs a division by 10. If we want to round, it would be (X/10)+0.5,
		; which can also be written as (X+5)/10

display_induct_uh
		movlw	.5
		addwf	xreg0, W			; Add 5 and move to yreg in one step
		movwf	yreg0
		movlw	0x00
		addwfc	xreg1, W
		movwf	yreg1
		movlw	0x00
		addwfc	xreg2, W
		movwf	yreg2
		movlw	0x00				; Very doubtful that this carries over,
		addwfc	xreg3, W			; but good practice to not rely on assumptions
		movwf	yreg3

		movlw	.10
		call	clear_x_to_W		; Prepare division by 10
		call	divyx
		call	print.x.10			; and print with one decimal

		; Finish up by printing 'uH' but with a real \mu

display_induct_units
		movlw	0xE4				; mu symbol on LCD
		call	LCD_write_data
		call	LCDwrstring
		data	"H   ", 0
		return




;***************************************************************
;
; Some numerical printing stuff

; Print signed integer in xreg

sprint.x
		btfss	xreg3, 7			; Sign bit set?
		bra		print.x				; No, just print positive integer
		movlw	A'-'				; Print the minus sign
		call	LCD_write_data
		call	chs_x				; Change sign of xreg
		bra		print.x				; and print positive integer

; Print x as a decimal number with 2 digits. Here, we begin with BCD
; conversion, then move xreg0 (two decimals) out of the way, print
; the remaining digits with leading-zero suppression as usual,
; followed by the decimals.

print.x.twodecs
		call	x_to_bcd
		movff	xreg0, xreg4		; Put these digits away for now
		movff	xreg1, xreg0
		movff	xreg2, xreg1
		movff	xreg3, xreg2
		call	printx				; no BCD conversion, but trailing-zero suppression
		movlw	A'.'
		call	LCD_write_data		; Print decimal point
		swapf	xreg4, W			; First decimal
		andlw	0x0F
		call	LCD_write_hex_nyb	; print nybble as ascii digit
		movf	xreg4, W			; Second decimal
		andlw	0x0F
		goto	LCD_write_hex_nyb	; print nybble as ascii digit


; print.x.10 prints a number in X with one decimal.
; The number in X must be 10 times the real number.

sprint.x.10
		btfss	xreg3, 7			; Sign bit set?
		bra		print.x.10			; No, just print positive integer
		movlw	A'-'				; Print the minus sign
		call	LCD_write_data
		call	chs_x				; Change sign of xreg

print.x.10

		call	copyxy
		call	clear_xreg
		movlw	.10
		movwf	xreg0
		call	divyx				; Divide by 10, remainder in Y
		call 	print.x				; Integer part of division
		movlw	A'.'
		call	LCD_write_data		; Print decimal point
		call	clear_xreg
		movff	yreg0,xreg0			; is only one digit anyway

print.x							; Display X in decimal

		call	x_to_bcd

printx							; Display hex X in LCD

		movf	xreg3,W				; since we suppress
		iorwf	xreg2,W				; leading zeroes,
		iorwf	xreg1,W				; display at least
		iorwf	xreg0,W				; one zero
		bnz		printx0				; if xreg==0
		movlw	0x00
		goto	LCD_write_hex_nyb

printx0							; Print hex xreg 3...0 in succession
		clrf	temp1				; temp1 is the leading zero flag
		movf	xreg3,W
		call	printx1
		movf	xreg2,W
		call	printx1
		movf	xreg1,W
		call	printx1
		movf	xreg0,W

printx1							; print whatever is in WREG
		movwf	temp2				; temp storage
		swapf	WREG				; high nybble first
		andlw	0x0f				; mask other part
		iorwf	temp1,F				; still leading zeroes?
		bnz		printx2				; No, print anyway
		tstfsz	WREG				; don't print if zero & leading
printx2
		call	LCD_write_hex_nyb	; print nybble

		movf	temp2,W				; Do the same with the low nybble
		andlw	0x0f
		iorwf	temp1,F
		bnz		printx3
		tstfsz	WREG
printx3
		goto	LCD_write_hex_nyb
		return



;***************************************************************
;
; Polled delays... put them here, no extra module needed.


; shortwait, a sequence of "dumb" NOPs

shortwait5
		rcall	shortwait		; about 1.2us
		rcall	shortwait
		rcall	shortwait
		rcall	shortwait		; with this, almost 5us, add another 1.2


shortwait			; NOP delay, leaves all registers intact
		nop			; One NOP takes 100ns (10 MIPS)
		nop			; so 10 NOPs are some 1.2 microseconds
		nop			; with call/return
		nop
		nop
		nop
		nop
		nop
		nop
		nop
		return


; Several TMR3-based polled delays. TMR3 runs at 5MHz, 200ns/tick

wait_10ms								; Fixed 10ms delay
		movlw	high (.65536 - .50000)
		movwf	TMR3H					; this is about as high as we get
		movlw	low (.65536 - .50000)	; with one shot of TMR3
		movwf	TMR3L
		bcf		PIR2, TMR3IF
		bra		wait_tmr3_rollover

wait_50us							; Fixed 50 microsecond delay
		movlw	high (.65536 - .250)
		movwf	TMR3H
		movlw	low (.65536 - .250)
		movwf	TMR3L
		bcf		PIR2, TMR3IF
		bra		wait_tmr3_rollover

wait250usec							; Fixed 250 microsecond delay
		movlw	high (.65536 - .1250)
		movwf	TMR3H
		movlw	low (.65536 - .1250)
		movwf	TMR3L
		bcf		PIR2, TMR3IF
		bra		wait_tmr3_rollover

wait_10us							; Fixed 10 microsecond delay
		movlw	high (.65536 - .50)
		movwf	TMR3H
		movlw	low (.65536 - .50)
		movwf	TMR3L
		bcf		PIR2, TMR3IF
		bra		wait_tmr3_rollover

; We also need 10usec, msec wait with a variable wait in WREG

wait250msec
		movlw	.250

waitmsec
		movwf	temp1
waitmsec_loop
		rcall	wait1msec
		decfsz	temp1
		bra		waitmsec_loop
		return

wait10usecp						; Wreg contains the wait time in units of 10usec
		mullw	.100			; Times 50 x 2, now twice as high as needed
		movlw	.13				; Overhead adjustment. Reduce delay
		subwf	PRODL, F		; by ~5 microseconds (MUST be an odd number for rounding)
		movlw	0x00
		subwfb	PRODH, F
		bcf		STATUS, C		; Divide by 2
		rrcf	PRODH, F		; by right-shifting
		rrcf	PRODL, F
		movlw	0xFF
		xorwf	PRODL, F		; one's complement
		xorwf	PRODH, F
		movlw	0x01			; add one to get two's complement
		addwf	PRODL, F
		movlw	0x00
		addwfc	PRODH, W		; needs intermediate storage
		movwf	TMR3H			; because TMR1H needs to be set first
		movf	PRODL, W
		movwf	TMR3L
		bcf		PIR2, TMR3IF	; Software-clear the rollover flag
		bra		wait_tmr3_rollover


wait1msec							; May be useful standalone, used in weitmsec.
		movlw	high (.65536 - .5000)
		movwf	TMR3H
		movlw	low (.65536 - .5000)
		movwf	TMR3L
		bcf		PIR2, TMR3IF

wait_tmr3_rollover
		nop
		btfss	PIR2, TMR3IF		; Wait for Timer1 to timeout (test bit 1 of PIR2)
		bra		wait_tmr3_rollover
		bcf		PIR2, TMR3IF		; Software-clear the flag
		return	












;***************************************************************
; 
;   ###     ###    ####            #   #  
;    #     #   #   #   #           #   #  
;    #     #       #   #           #   #  
;    #      ###    ####    #####   #####  
;    #         #   # #             #   #  
;    #     #   #   #  #            #   #  
;   ###     ###    #   #           #   #  
; 
;  *** HIGH-PRIORITY INTERRUPTS ***
;***************************************************************


; High priprity interrupt
; =======================
;
; This intrp uses the fast register stack, so WREG, STATUS,
; and BSR are saved during interrupt response. 
; Also, further interrupts are disabled at this point
; By exiting with retfie fast, we automatically restore
; WREG, STATUS, and BSR and re-enable GIE.


Intrp_ent
		btfss	INTCON, TMR0IF			; TMR0 interrupt?
		bra		intr_tmr0_exit			; no, check next source

; TMR0 interrupt. Reload the timer and ...


		movlw	high TMR0PRELOAD
		movwf	TMR0H
		movlw	low  TMR0PRELOAD
		movwf	TMR0L					; Ready to roll again
		bcf		INTCON, TMR0IF			; and software-clear the interrupt condition


		; To test if this is a valid oscillation, we kick-start the ADC
		; (and do this every 10ms)

		movlw	B'00001101'				; Still Channel 3
		movwf	ADCON0
		bsf		LATC, LED				; LED is on, dimly (reset in ADC intr)
		bsf		ADCON0, GO
		bcf		PIR1, ADIF
		bsf		PIE1, ADIE

		movf	tencounter, W			; Run down to zero?
		bz		intr_tmr0_200ms			; then we have a 200ms event
		decfsz	tencounter, F			; Not yet zero, count it down
		bra		intr_tmr0_exit

		; Counter-to-20 elapsed (i.e., 200ms elapsed). Read counter.

intr_tmr0_200ms
		movlw	.20
		movwf	tencounter				; Reset the counter-to-ten
		movf	TMR1L, W				; Read-out of oscillator counter
		movwf	count_l
		movf	TMR1H, W
		movwf	count_h
		movf	tmr1ovfl, W				; Let's be consiatent in ISR: Full 24 bits
		movwf	count_u
		clrf	TMR1H					; and start over with that timer
		clrf	TMR1L					; (mind the H-L order)
		clrf	tmr1ovfl				; Clear the overflow byte, too
		bcf		PIR1, TMR1IF
		bsf		runstatus, RS_DAVAIL


intr_tmr0_exit

;-----------------------------------------
; TMR1: This occurs when TMR1 overflows

		btfss	PIR1, TMR1IF
		bra		intr_tmr1_exit
		bcf		PIR1, TMR1IF

		incf	tmr1ovfl				; increment u byte. This makes TMR1 effectively
										; a 24-bit counter, although we shouldn't need
										; this byte. Maybe indicate overflow?

intr_tmr1_exit


;-----------------------------------------
; ADIF: A/D conversion interrupt

		btfss	PIR1, ADIF
		bra		intr_adif_exit
		bcf		PIR1, ADIF
		bcf		PIE1, ADIE			; Keep this a one-shot interrupt

		; Result in ADRES<L:H>, shift into osc_lvl_<l:h> via simple IIR

		movf	ADRESL, W
		addwf	osc_lvl_l, F
		movf	ADRESH, W
		addwfc	osc_lvl_h, F
		bcf		STATUS, C
		rrcf	osc_lvl_h, F
		rrcf	osc_lvl_l, F

		; Test for low amplitude levels, decide RS_CONNECTED on that
		; Unfortunately, this is iffy, because it depends on the AGC potentiometer

		bsf		runstatus, RS_CONNECTED
		tstfsz	osc_lvl_h
		bra		intr_adif_exit
		movlw	.25
		cpfsgt	osc_lvl_l					; Low byte larger than 20? Or 98mV?
		bcf		runstatus, RS_CONNECTED		; If not, probably nothing connected

intr_adif_exit
		btfss	runstatus, RS_CONNECTED		; Leave LED on if level is good, otherwise
		bcf		LATC, LED					; ...turn off the LED so that it appears dim



int_fast_exit
		retfie	FAST			; Ret fast, load W, Status, BSR from shadow regs



;***************************************************************
; 
;   ###     ###    ####            #      
;    #     #   #   #   #           #      
;    #     #       #   #           #      
;    #      ###    ####    #####   #      
;    #         #   # #             #      
;    #     #   #   #  #            #      
;   ###     ###    #   #           #####  
; 
;
;  *** LOW-PRIORITY INTERRUPTS ***
;*************************************************************************

; The low-priority interrupt cannot automatically use the
; fast register stack, so we need to explicitly save the 
; registers and re-enable GIE and PEIE



Low_intrp_ent

		movwf	iwreg			; Save the registers
		movff	STATUS,istatus
		movff	BSR,ibsr




;-------------------------------------------------------
;
; END low-priority interrupt chain

low_intr_exit
		movff	ibsr,BSR
		movf	iwreg,W
		movff	istatus,STATUS
		bsf		INTCON,PEIE		; re-enable lo-pri interrupts
		bsf		INTCON,GIEH
		retfie					; Cannot use FAST in low-pri interrupt serivce

;*********************************************************************






		END				; that's all for now, folks.

;*********************************************************************





